/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "couponexportdialog.h"
#include "couponcolumnselectiondialog.h"
#include <QFileDialog>
#include <QMessageBox>
#include <QPushButton>
#include <QTableView>
#include <QTextStream>
#include <QVBoxLayout>

CouponExportDialog::CouponExportDialog(QSqlTableModel *model, QWidget *parent)
    : QDialog(parent)
    , m_model(model)
    , view(new QTableView(this))
{

    m_model->select();

    view->setModel(m_model);

    QPushButton *exportButton = new QPushButton("Export to CSV", this);
    connect(exportButton, &QPushButton::clicked, this, &CouponExportDialog::exportToCSV);

    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(view);
    layout->addWidget(exportButton);

    setLayout(layout);
}

CouponExportDialog::~CouponExportDialog()
{
    // Cleanup, if necessary
}

void CouponExportDialog::exportToCSV()
{
    CouponColumnSelectionDialog dialog(m_model, this);
    if (dialog.exec() == QDialog::Accepted) {
        QStringList selectedColumns = dialog.selectedColumns();
        QString fileName = QFileDialog::getSaveFileName(this, "Save as CSV", "", "CSV Files (*.csv)");
        if (!fileName.isEmpty()) {
            if (writeCSV(fileName, selectedColumns)) {
                QMessageBox::information(this, "Success", "Data exported successfully.");
            } else {
                QMessageBox::critical(this, "Error", "Failed to export data.");
            }
        }
    }
}

bool CouponExportDialog::writeCSV(const QString &fileName, const QStringList &columns)
{
    QFile file(fileName);
    if (!file.open(QIODevice::WriteOnly)) {
        return false;
    }

    QTextStream out(&file);
    // Write header
    for (int i = 0; i < columns.size(); ++i) {
        if (i > 0) out << ",";
        out << columns.at(i);
    }
    out << "\n";

    // Write data
    for (int row = 0; row < m_model->rowCount(); ++row) {
        for (int col = 0; col < columns.size(); ++col) {
            if (col > 0) out << ",";
            int columnIndex = m_model->fieldIndex(columns.at(col));
            out << m_model->data(m_model->index(row, columnIndex)).toString();
        }
        out << "\n";
    }

    file.close();
    return true;
}
