/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "pushbutton.h"
#include "preferences/settings.h"

#include <QGuiApplication>
#include <QScreen>

PushButton::PushButton(QWidget *parent)
    : QPushButton(parent)
{
    initialize();
}

PushButton::PushButton(const QIcon &icon, const QString &text, QWidget *parent)
    : QPushButton(icon, text, parent)
{
    setIconSize(QSize(32, 32));
    initialize();
}

PushButton::PushButton(const QString &text, QWidget *parent)
    : QPushButton(text, parent)
{
    initialize();
}

PushButton::~PushButton() = default;

// Getter und Setter für die Button-Größe mit Q_PROPERTY-Unterstützung
QSize PushButton::buttonSize() const
{
    return m_buttonSize;
}

void PushButton::setButtonSize(const QSize &size)
{
    if (m_buttonSize != size) {
        m_buttonSize = size;
        applySize();
        emit buttonSizeChanged(m_buttonSize);
    }
}

void PushButton::initialize()
{
    Settings settings;

    // Standardgröße lesen und Skalierung anwenden
    QSize defaultSize = settings.value("ButtonSize", QSize(150, 60)).toSize();
    m_buttonSize = applyDpiScaling(defaultSize);

    applySize();

    // Stylesheet hinzufügen
    // setStyleSheet("PushButton { border-radius: 5px; }");
}

void PushButton::applySize()
{
    setFixedHeight(m_buttonSize.height());
    setMinimumWidth(m_buttonSize.width());
}

QSize PushButton::applyDpiScaling(const QSize &size) const
{
    qreal dpiScale = QGuiApplication::primaryScreen()->logicalDotsPerInch() / 96.0;
    return QSize(size.width() * dpiScale, size.height() * dpiScale);
}
