#include "qrkprinter.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "database.h"

#include <QElapsedTimer>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonValue>
#include <QMap>
#include <QPrinterInfo>
#include <QSqlDatabase>

#include <QVariant>

#include <QDebug>

// Statische Map, die physische Drucker mit spezifischen Definitionen speichert
static QMap<QString, QPrinter *> globalPhysicalPrinterCache;
static QMap<QString, QList<QPrinter *> > globalPrinterMap;

QRKPrinter::QRKPrinter(QString printername, QObject *parent)
    : QObject(parent)
    , m_printername(printername)
{
    initPrinters();
}

QRKPrinter::QRKPrinter(int printerid, QObject *parent)
    : QObject(parent)
    , m_printername(Database::getPrinterName(printerid))
{
    initPrinters();
}

QRKPrinter::QRKPrinter(QSplashScreen *splash, QObject *parent)
    : QObject(parent)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT name FROM printers");
    query.exec();
    while (query.next()) {
        QString name = query.value("name").toString();
        if (splash)
            splash->showMessage(QObject::tr("Initialisiere Drucker für %1 ...").arg(name),
                Qt::AlignLeft | Qt::AlignBottom, Qt::darkYellow);

        qDebug() << "Function Name: " << Q_FUNC_INFO << "initPrinter: " << name;
        m_printername = name;
        initPrinters();
    }
}

QRKPrinter::~QRKPrinter()
{
    if (m_printers == &globalPrinterMap[m_printername]) {
        // Die Liste wird nicht gelöscht, da sie in der Map verwaltet wird
        return;
    }

    while (!m_printers->isEmpty()) {
        delete m_printers->takeFirst();
    }
    delete m_printers;
    m_printers = Q_NULLPTR;
}

void QRKPrinter::initPrinters()
{
    QElapsedTimer timer;
    timer.start();

    // Prüfen, ob bereits eine Liste für den Druckernamen existiert
    if (globalPrinterMap.contains(m_printername) && !globalPrinterMap[m_printername].isEmpty()) {
        m_printers = &globalPrinterMap[m_printername];
        qint64 elapsed = timer.nsecsElapsed();
        qDebug() << "Function Name: " << Q_FUNC_INFO << "initPrinters: " << m_printers->size() << " in " << elapsed
                 << "ns";
        return;
    }

    // Initialisierung, falls noch keine Liste für diesen Druckernamen existiert
    QList<QPrinter *> printerList;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT printer FROM printers WHERE name=:name");
    query.bindValue(":name", m_printername);
    query.exec();
    int printers = 0;
    while (query.next()) {
        QByteArray data = QByteArray::fromBase64(query.value("printer").toByteArray());
        QJsonArray jArray = QJsonDocument().fromJson(data).array();

        CSqlQuery definition(dbc, Q_FUNC_INFO);
        definition.prepare("SELECT definition FROM printerdefs WHERE id=:id");
        foreach (const QJsonValue &value, jArray) {
            QJsonObject obj = value.toObject();
            QString physicalPrinterName = obj["name"].toString();
            int definitionId = obj["definitionid"].toInt();

            // Erstellen des Schlüssels für die Cache-Map, der den Druckernamen und die Definition eindeutig
            // identifiziert
            QString cacheKey = physicalPrinterName + "_" + QString::number(definitionId);

            QPrinter *printer;
            if (globalPhysicalPrinterCache.contains(cacheKey)) {
                // Verwenden des vorhandenen Druckers aus dem Cache
                printer = globalPhysicalPrinterCache[cacheKey];
                qDebug() << "Function Name: " << Q_FUNC_INFO << "initPrinter " << m_printername
                         << " from cache: " << cacheKey;
            } else {
                // Neuer Drucker, da Kombination nicht im Cache
                qDebug() << "Function Name: " << Q_FUNC_INFO << "initPrinter " << m_printername
                         << " not in cache: " << cacheKey;
                printer = new QPrinter;
                if (physicalPrinterName.contains("QrkPDF")) {
                    printer->setOutputFormat(QPrinter::OutputFormat::PdfFormat);
                } else {
                    printer->setPrinterName(physicalPrinterName);
                }
                definition.bindValue(":id", definitionId);
                definition.exec();
                if (definition.next()) {
                    QByteArray data = QByteArray::fromBase64(definition.value("definition").toByteArray());
                    setDefinition(QJsonDocument().fromJson(data).object(), printer);
                    globalPhysicalPrinterCache[cacheKey] = printer; // Speichern im Cache
                }
            }
            printerList.append(printer);
            printers++;
        }
    }

    // Speichern der neuen Liste in der Map
    globalPrinterMap[m_printername] = printerList;
    m_printers = &globalPrinterMap[m_printername];

    qint64 elapsed = timer.nsecsElapsed();
    qDebug() << "Function Name: " << Q_FUNC_INFO << "initPrinters: " << printers << " in " << elapsed << "ns";

    if (printerList.isEmpty()) {
        setDefaultPDFPrinter();
    }
}

void QRKPrinter::setDefaultPDFPrinter()
{
    QPrinter *printer = new QPrinter;
    printer->setFullPage(true);
    printer->setOutputFormat(QPrinter::OutputFormat::PdfFormat);
    printer->setPageSize(QPageSize(QPageSize::A4));
    printer->setFullPage(false);
    globalPrinterMap[m_printername].append(printer);
    m_printers = &globalPrinterMap[m_printername];
}

void QRKPrinter::setDefinition(const QJsonObject &object, QPrinter *&printer)
{
    printer->setFullPage(true);


    QPrinterInfo pInfo = QPrinterInfo::printerInfo(printer->printerName());

    qDebug() << "Function Name: " << Q_FUNC_INFO << "Printer outputformat: " << printer->outputFormat() << "("
             << (QPrinter::OutputFormat::PdfFormat == printer->outputFormat()) << ") printername; "
             << pInfo.printerName();

    QList<QPageSize> pSizes = pInfo.supportedPageSizes();


    bool pdf = (QPrinter::OutputFormat::PdfFormat == printer->outputFormat());
    bool hasCustomFormat = false;
    if (pdf) {
        hasCustomFormat = pdf;
    } else {
        QListIterator<QPageSize> iter(pSizes);
        while (iter.hasNext()) {
            QPageSize pageSize = iter.next();
            if (pageSize.id() == static_cast<int>(QPrinter::Custom)) {
                hasCustomFormat = true;
                break;
            }
        }
    }

    if (object["type"].toString() == "custom" && hasCustomFormat) {
        if (pdf) {
            QSizeF sizeF(object["paperWidth"].toInt(), 297);
            printer->setPageSize(QPageSize(sizeF, QPageSize::Millimeter));
            qDebug() << "Function Name: " << Q_FUNC_INFO << "PDF Printer: true";
            qDebug() << "Function Name: " << Q_FUNC_INFO << "Printer set Format: " << sizeF;
        } else {
            QSizeF sizeF(object["paperWidth"].toInt(), object["paperHeight"].toInt());
            printer->setPageSize(QPageSize(sizeF, QPageSize::Millimeter));
            qDebug() << "Function Name: " << Q_FUNC_INFO << "Printer has Custom Format: true";
            qDebug() << "Function Name: " << Q_FUNC_INFO
                     << "Printer set Format: " << QPageSize(sizeF, QPageSize::Millimeter);
        }
    } else if (object["type"].toString() == "A4" && pSizes.contains(QPageSize(QPageSize::A4))) {
        printer->setPageSize(QPageSize(QPageSize::A4));
    } else if (object["type"].toString() == "A5" && pSizes.contains(QPageSize(QPageSize::A5))) {
        printer->setPageSize(QPageSize(QPageSize::A5));
    } else {
        printer->setPageSize(pInfo.defaultPageSize());
        qDebug() << "Function Name: " << Q_FUNC_INFO << "Printer has Custom Format: false";
        qDebug() << "Function Name: " << Q_FUNC_INFO << "Printer set Format: " << pInfo.defaultPageSize();
    }

    const QMarginsF marginsF(object["marginLeft"].toDouble(), object["marginTop"].toDouble(),
        object["marginRight"].toDouble(), object["marginBottom"].toDouble());

    printer->setPageMargins(marginsF, QPageLayout::Millimeter);
    printer->setFullPage(false);
}

QList<QPrinter *> QRKPrinter::getPrinterList()
{
    QList<QPrinter *> copy;
    if (m_printers) {
        copy = *m_printers;
    }
    return copy;
}

void QRKPrinter::clearGlobalPrinterList(const QString &printerName)
{
    if (globalPrinterMap.contains(printerName)) {
        globalPrinterMap.remove(printerName);
    }
}

void QRKPrinter::clearAllGlobalResources()
{
    for (auto it = globalPhysicalPrinterCache.begin(); it != globalPhysicalPrinterCache.end(); ++it) {
        QString key = it.key();
        QPrinter *printer = it.value();

        qInfo() << "Function Name: " << Q_FUNC_INFO << "Remove Printer with key: " << key;
        delete printer;
    }

    globalPhysicalPrinterCache.clear();
    globalPrinterMap.clear();
}
