#include "backupworker.h"

#include "JlCompress.h"
#include "backup.h"
#include "csqlquery.h"
#include "database.h"
#include "qrksettings.h"

#include <QApplication>
#include <QElapsedTimer>
#include <QSqlDatabase>
#include <QSqlError>
#include <QStandardPaths>

#include <QDebug>

BackupWorker::BackupWorker(
    const QStringList &fullpathFilelist, const QString &backupDir, const QString &tempname, const QString &confname)
    : m_fullpathFilelist(fullpathFilelist)
    , m_backupDir(backupDir)
    , m_tempname(tempname)
    , m_confname(confname)
{
}

BackupWorker::~BackupWorker()
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << "processBackup completed successfully.";
}

void BackupWorker::run()
{
    qInfo() << "Function Name: " << Q_FUNC_INFO << "copyFiles started.";
    bool copySuccess = copyDatabaseFiles(m_fullpathFilelist, m_backupDir, m_tempname);
    if (!copySuccess) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Datacopy failed!";
        emit finishedCopyFiles();
        return;
    }
    emit finishedCopyFiles();

    qInfo() << "Function Name: " << Q_FUNC_INFO << "processBackup started.";

    QElapsedTimer t;
    t.start();

    QString outfile
        = QString("%1/data_%2.zip").arg(m_backupDir).arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));

    QStringList fullpathFilelist;
    fullpathFilelist << QString(m_backupDir + "/qrktmp/" + m_tempname + "/%1-%2%3.db")
                            .arg(QDate::currentDate().year())
                            .arg(qApp->applicationName())
                            .arg(m_confname);

    QString filename;
    exportTables(m_backupDir + "/qrktmp/" + m_tempname, m_confname, filename);
    fullpathFilelist.append(filename);

    bool ok = JlCompress::compressFiles(outfile, fullpathFilelist);
    if (!ok) qWarning() << "Function Name: " << Q_FUNC_INFO << " JlCompress::compressFile:" << ok;

    removeOldestFiles();
    QFile::remove(filename);

    qDebug() << "Function Name: " << Q_FUNC_INFO << "Total Time elapsed: " << t.elapsed() << " ms";
    removeDir(m_backupDir + "/qrktmp/" + m_tempname);
}

bool BackupWorker::removeDir(const QString &dirName)
{
    bool result = true;
    QDir dir(dirName);

    if (dir.exists(dirName)) {
        Q_FOREACH (QFileInfo info,
            dir.entryInfoList(
                QDir::NoDotAndDotDot | QDir::System | QDir::Hidden | QDir::AllDirs | QDir::Files, QDir::DirsFirst)) {
            if (info.isDir()) {
                result = removeDir(info.absoluteFilePath());
            } else {
                result = QFile::remove(info.absoluteFilePath());
            }

            if (!result) {
                return result;
            }
        }
        result = dir.rmdir(dirName);
    }
    return result;
}

/*
void BackupWorker::exportTables(const QString &backupDir, const QString &configName, QString &filename)
{
    QDate date = QDate::currentDate();
    QrkSettings settings;
    QFileInfo fi(settings.fileName());
    QString basename = fi.baseName();
    QString databasefile = QString(backupDir + "/%1-%2.db").arg(date.year()).arg(basename);

    if (QFile::exists(databasefile)) {
        QSqlDatabase dbc;
        if (dbc.isOpen()) dbc.close();

        dbc = QSqlDatabase::addDatabase("QSQLITE", m_tempname);
        dbc.setDatabaseName(databasefile);
        dbc.open();
        CSqlQuery testquery(dbc, Q_FUNC_INFO);
        testquery.prepare(QString("VACUUM;"));
        if (!testquery.exec()) {
            dbc.close();
            return;
        }

        QHash<QString, QStringList> tablesWithFields; // It holds the table name and its fields
        QStringList tables = dbc.tables();
        CSqlQuery query(dbc, Q_FUNC_INFO);

        bool sqlite = (dbc.driverName() == "QSQLITE");
        foreach (QString table, tables) {

            QStringList fields;
            if (sqlite) {
                query.exec(QString("PRAGMA TABLE_INFO(%1)").arg(table));
                while (query.next()) {
                    fields << query.value(1).toString();
                }
            } else {
                query.exec(QString("DESCRIBE %1;").arg(table));
                while (query.next()) {
                    fields << query.value(0).toString();
                }
            }
            tablesWithFields.insert(table, fields);
        }

        filename = QString("%1/%2%3_dump_%4.sql")
                       .arg(backupDir)
                       .arg(qApp->applicationName())
                       .arg(configName)
                       .arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));
        QFile f(filename);
        f.open(QIODevice::Append | QIODevice::Text);
        QTextStream streamer(&f);

        // If constraints can't be dropped in the target database, some reordering of
        // the INSERT statements may be needed
        QStringList sortedTables = tablesWithFields.keys();
        // sortedTables.move(sorted.indexOf("table1"),0);
        // ...
        (sqlite) ? streamer << "BEGIN TRANSACTION;\n" : streamer << "SET FOREIGN_KEY_CHECKS=0;\nSTART TRANSACTION;\n";
        foreach (const QString &table, sortedTables) {
            qApp->processEvents();
            if (table == "sqlite_sequence") continue;

            streamer << getCreateTable(table) + "\n";

            QString statement = QString("INSERT INTO %1 VALUES(\n'").arg(table);
            QStringList fields = tablesWithFields.value(table);
            QString fieldsString = fields.join(",");
            query.exec(QString("SELECT %1 FROM %2").arg(fieldsString).arg(table));
            if (!query.next()) continue;

            query.previous();
            while (query.next()) {
                qApp->processEvents();
                for (int i = 0; i < fields.size(); ++i) {
                    QString value = query.value(i).toString();
                    value.replace("'", "''"); // Handle single quotes inside strings
                    if (value.isNull()) {
                        value = "NULL";
                        statement.chop(1); // NULL should not appear inside quotes
                        statement.append(value + ",'");
                    } else {
                        statement.append(value + "','");
                    }
                    qApp->processEvents();
                }
                statement.chop(2);          // Remove comma and single quote from the end of value group
                statement.append("),\n('"); // Close the value group and start a new one
            }
            statement.chop(4);              // Remove comma, opening parenthesis and single quote from the end
            streamer << statement << ";\n"; // Complete the INSERT statement
        }
        (sqlite) ? streamer << "COMMIT;" : streamer << "COMMIT;\nSET FOREIGN_KEY_CHECKS=1;";
        f.close();
        dbc.close();
    }
}
*/

void BackupWorker::exportTables(const QString &backupDir, const QString &configName, QString &filename)
{
    QDate date = QDate::currentDate();
    QrkSettings settings;
    QFileInfo fi(settings.fileName());
    QString basename = fi.baseName();
    QString databasefile = QString("%1/%2-%3.db").arg(backupDir).arg(date.year()).arg(basename);

    if (QFile::exists(databasefile)) {
        QSqlDatabase dbc;
        if (dbc.isOpen()) dbc.close();

        dbc = QSqlDatabase::addDatabase("QSQLITE", m_tempname);
        dbc.setDatabaseName(databasefile);
        dbc.open();
        CSqlQuery query(dbc, Q_FUNC_INFO);

        QStringList tables = dbc.tables();
        filename = QString("%1/%2%3_dump_%4.sql")
                       .arg(backupDir)
                       .arg(qApp->applicationName())
                       .arg(configName)
                       .arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));
        QFile file(filename);
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
            qWarning() << "Unable to open file for writing:" << filename;
            return;
        }

        QTextStream streamer(&file);
        streamer.setCodec("UTF-8");

        bool sqlite = (dbc.driverName() == "QSQLITE");
        (sqlite) ? streamer << "BEGIN TRANSACTION;\n" : streamer << "SET FOREIGN_KEY_CHECKS=0;\nSTART TRANSACTION;\n";

        foreach (const QString &table, tables) {
            if (table == "sqlite_sequence") continue;

            qApp->processEvents();
            streamer << getCreateTable(table) << "\n";

            QStringList fields;
            if (sqlite) {
                query.exec(QString("PRAGMA TABLE_INFO(%1)").arg(table));
                while (query.next()) {
                    fields << query.value(1).toString();
                }
            } else {
                query.exec(QString("DESCRIBE %1;").arg(table));
                while (query.next()) {
                    fields << query.value(0).toString();
                }
            }

            QString selectQuery = QString("SELECT %1 FROM %2").arg(fields.join(",")).arg(table);
            query.exec(selectQuery);

            while (query.next()) {
                qApp->processEvents();
                QStringList values;

                for (int i = 0; i < fields.size(); ++i) {
                    QVariant value = query.value(i);

                    if (value.isNull()) {
                        values << "NULL"; // NULL ohne Quotes
                    } else if (value.type() == QVariant::Int || value.type() == QVariant::Double
                        || value.type() == QVariant::LongLong) {
                        values << value.toString(); // Numerische Werte direkt einfügen
                    } else if (value.type() == QVariant::Bool) {
                        values << (value.toBool() ? "1" : "0"); // Boolean als 1 oder 0
                    } else if (value.type() == QVariant::Date) {
                        values << QString("'%1'").arg(value.toDate().toString("yyyy-MM-dd"));
                    } else if (value.type() == QVariant::DateTime) {
                        values << QString("'%1'").arg(value.toDateTime().toString("yyyy-MM-dd HH:mm:ss"));
                    } else if (value.type() == QVariant::Time) {
                        values << QString("'%1'").arg(value.toTime().toString("HH:mm:ss"));
                    } else {
                        QString strValue = value.toString().replace("'", "''"); // Escaping für Strings
                        values << QString("'%1'").arg(strValue);
                    }
                }

                streamer << QString("INSERT INTO %1 VALUES(%2);\n").arg(table).arg(values.join(", "));
            }
        }

        (sqlite) ? streamer << "COMMIT;" : streamer << "COMMIT;\nSET FOREIGN_KEY_CHECKS=1;";
        file.close();
        dbc.close();
    }
}

void BackupWorker::removeOldestFiles()
{
    QElapsedTimer t;
    t.start();
    QrkSettings settings;
    int keep = settings.value("keepMaxBackups", -1).toInt();
    if (keep == -1) return;

    QString backupDir = QDir(
        settings.value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation)).toString())
                            .absolutePath();

    QDir dir(backupDir);
    dir.setNameFilters(QStringList() << "*.zip");
    dir.setFilter(QDir::Files);
    dir.setSorting(QDir::Time | QDir::Reversed);
    QStringList list = dir.entryList();
    while (list.size() > keep) {
        qApp->processEvents();
        QString f = list.takeFirst();
        QFile::remove(dir.absoluteFilePath(f));
        qInfo() << "Function Name: " << Q_FUNC_INFO << " Remove old backup FileName: " << f;
    }
    qDebug() << "Function Name: " << Q_FUNC_INFO << "Time elapsed: " << t.elapsed() << " ms";
}

QString BackupWorker::getCreateTable(const QString &tablename)
{
    /*
    "SHOW CREATE TABLE `users`;"
    "SELECT sql FROM sqlite_master WHERE name='users'"

    "DROP TABLE [IF EXISTS] table_name"
    */
    QString drop = QString("\nDROP TABLE IF EXISTS `%1`;\n").arg(tablename);

    QSqlDatabase dbc = Database::database();

    CSqlQuery query(dbc, Q_FUNC_INFO);
    if (dbc.driverName() == "QSQLITE") {
        query.prepare("SELECT sql FROM sqlite_master WHERE name=:tablename;");
        query.bindValue(":tablename", tablename);
        if (query.exec())
            if (query.next()) return drop + query.value("sql").toString() + ";\n";
    } else {
        if (query.exec(QString("SHOW CREATE TABLE %1;").arg(tablename)))
            if (query.next()) return drop + query.value(1).toString() + ";\n";
    }

    qDebug() << "Function Name: " << Q_FUNC_INFO << " lasterror:" << query.lastError().text();
    qDebug() << "Function Name: " << Q_FUNC_INFO << " lastquery:" << Database::getLastExecutedQuery(query);

    return "";
}

bool BackupWorker::copyDatabaseFiles(
    const QStringList &filePaths, const QString &destinationDir, const QString &tempname)
{
    QElapsedTimer t;
    t.start();

    QDir destDir(destinationDir + "/qrktmp/" + tempname);

    // Prüfen, ob das Zielverzeichnis existiert, falls nicht, erstellen
    if (!destDir.exists() && !destDir.mkpath(".")) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Can not create Destinationdir: " << destinationDir;
        return false;
    }

    for (const QString &filePath : filePaths) {
        QFileInfo fileInfo(filePath);
        if (!fileInfo.exists() || !fileInfo.isFile()) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << "File not exists:" << filePath;
            continue;
        }

        QString destFilePath = destDir.filePath(fileInfo.fileName());

        if (!QFile::copy(filePath, destFilePath)) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << "Can not copy File: " << filePath << " to "
                       << destFilePath;
            continue;
        }
        qApp->processEvents();
    }
    qDebug() << "copyDatabaseFiles Time elapsed:" << t.elapsed() << " ms";

    return true;
}
