/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "profeatures.h"
#include "3rdparty/ckvsoft/preferences/registrationtab.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "database.h"
#include "defines.h"
#include "givendialog.h"
#include "pluginmanager/Interfaces/independentinterface.h"
#include "pluginmanager/pluginmanager.h"
#include "profeaturesprint.h"
#include "qrksettings.h"

#include <QApplication>
#include <QCheckBox>
#include <QComboBox>
#include <QDateTime>
#include <QFile>
#include <QMap>
#include <QSqlError>
#include <QSqlQueryModel>
#include <QTime>

QMap<QString, QMap<QDateTime, bool> > s_profeatures;

bool ProFeatures::isValid()
{

    if (RegistrationTab::isActive(qApp->property("appBaseName").toString())) {
        bool value = false;
        if (ProFeatures::getProfeatures(value)) return value;
        qDebug() << "Function Name: " << Q_FUNC_INFO << " Start: " << QTime::currentTime();
        RegistrationTab reg(qApp->property("appBaseName").toString(), false);
        int days;
        bool valid = reg.isValid(days);
        ProFeatures::setProfeatures(valid);
        qDebug() << "Function Name: " << Q_FUNC_INFO << "  End: " << QTime::currentTime();
        return valid;
    }

    return false;
}

void ProFeatures::setProfeatures(bool value)
{
    QDateTime t(QDate::currentDate(), QTime(23, 23, 59));
    QMap<QDateTime, bool> qm;
    qm.insert(t, value);
    s_profeatures.insert("ProFeatures", qm);
}

bool ProFeatures::getProfeatures(bool &value)
{
    if (s_profeatures.contains("ProFeatures")) {
        QMap<QDateTime, bool> qm = s_profeatures.first();
        QDateTime t = qm.firstKey();
        qint64 diff = QDateTime::currentDateTime().secsTo(t);
        value = qm.first();
        if (diff >= 0) return true;
    }
    return false;
}

double ProFeatures::GroupEditFeatures(QSqlQueryModel *model, int index, bool &valid)
{
    if (index == 0 || !ProFeatures::isValid()) {
        valid = false;
        return {};
    }

    valid = true;
    qDebug() << "Function Name: " << Q_FUNC_INFO << "idx " << index << " value " << model->data(model->index(index, 1));
    return model->data(model->index(index, 1)).toDouble();
}

void ProFeatures::SettingsFeatures(QCheckBox *box, const QString what)
{
    QrkSettings settings;
    if (ProFeatures::isValid()) {
        box->setChecked(settings.value(what, false).toBool());
    } else {
        QString text = tr("Nur in der PRO Version verfügbar");
        box->setEnabled(false);
        box->setStatusTip(text);
    }
}

void ProFeatures::SettingsFeatures(QComboBox *box, const QString what)
{
    Q_UNUSED(what)
    box->setEnabled(ProFeatures::isValid());
}

QString ProFeatures::VersionInfo()
{
    if (ProFeatures::isValid()) return qApp->property("appBaseName").toString();

    return qApp->applicationName();
}

// Database stuff
double ProFeatures::getTaxFromProduct(const int id)
{
    bool feature = ProFeatures::isValid();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery groupquery(dbc, Q_FUNC_INFO);
    double tax = Database::getDefaultTax().toDouble();

    query.prepare("SELECT tax, groupid FROM products WHERE id = :id");
    query.bindValue(":id", id);
    if (query.exec()) {
        if (query.next()) {
            tax = query.value("tax").toDouble();
            if (!feature) return tax;

            groupquery.prepare("SELECT tax FROM groups WHERE id = :groupid");
            groupquery.bindValue(":groupid", query.value("groupid").toInt());
            if (groupquery.exec()) {
                if (groupquery.next()) {
                    QString grouptax = groupquery.value("tax").toString();
                    if (!grouptax.isEmpty()) tax = grouptax.toDouble();
                }
            }
        }
    }

    return tax;
}

bool ProFeatures::getTaxFromGroup(const QString &groupname, QBCMath &tax)
{
    if (ProFeatures::isValid()) {
        QSqlDatabase dbc = Database::database();
        CSqlQuery groupquery(dbc, Q_FUNC_INFO);
        groupquery.prepare("SELECT tax FROM groups WHERE name = :groupname");
        groupquery.bindValue(":groupname", groupname);
        if (groupquery.exec()) {
            if (groupquery.next()) {
                QString grouptax = groupquery.value("tax").toString();
                if (!grouptax.isEmpty()) {
                    tax = grouptax.toDouble();
                    return true;
                }
            }
        }
    }

    return false;
}

QString ProFeatures::getCheckoutContents(bool beautify)
{
    QVariant value = "";
    QString strValue = "";
    AbstractDataBase::select_globals("cashbook_begin", value, strValue);

    if (!value.isValid() && strValue.isEmpty()) return {};

    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("select sum(gross) AS total FROM cashbook where deleted = 0");
    // query.prepare("select sum(gross) AS total FROM cashbook");
    bool ok = query.exec();
    if (!ok) {
        return {};
    }

    QBCMath total(value.toLongLong());
    if (query.next()) {
        total += (query.value("total").toLongLong());
    }

    if (beautify) {
        total /= 100;
        total.round(2);
        return total.toLocale() + " " + Database::getShortCurrency();
    } else {
        return total.getIntPart();
    }
}

bool ProFeatures::CouponDialog(ReceiptItemModel &model, QBCMath &sum, QBCMath &redeem, QString &code, bool &isSingle)
{

    IndependentInterface *couponplugin
        = qobject_cast<IndependentInterface *>(PluginManager::instance()->getObjectByName("Coupon"));
    if (couponplugin && couponplugin->isActivated()) {
        QrkCustomDialog *dialog = couponplugin->CustomDialog(&model);
        dialog->setValue("sum", sum.toString());
        if (dialog->exec() == 0) {
            couponplugin->deleteLater();
            return false;
        }
        redeem = dialog->getValue("redeem");
        code = dialog->getValue("code");
        redeem.round(2);
        isSingle = dialog->getValue("single").toInt();
        if (isSingle) {
            QString tax = dialog->getValue("tax");
            if (redeem > 0) {
                model.plus();
                int row = model.rowCount() - 1;
                model.setItem(row, REGISTER_COL_PRODUCT, new QStandardItem(QString("Gutschein %1").arg(code)));
                model.setItem(row, REGISTER_COL_TAX, new QStandardItem(tax));
                model.setItem(row, REGISTER_COL_SINGLE, new QStandardItem((redeem * -1).toString()));
            }
        } else if (((sum - redeem).toDouble() != 0.0)) {
            double s = (sum - redeem).toDouble();
            GivenDialog given(s);
            given.setCoupon((sum - redeem).toDouble());
            if (given.exec() == 0) {
                couponplugin->deleteLater();
                return false;
            }
            QMap<int, double> map = given.getGiven();
            if (map.size() > 1) {
                map.remove(PAYED_BY_CASH);
                map.insert(PAYED_BY_COUPON, redeem.toDouble());
            } else {
                double g = map.take(PAYED_BY_CASH);
                map.insert(PAYED_BY_CASH, g > (sum - redeem).toDouble() ? g : (sum - redeem).toDouble());
            }
            map.insert(PAYED_BY_COUPON, redeem.toDouble());
            model.setGiven(map);
            // m_receiptitemmodel->setGiven(PAYED_BY_COUPON, redeem.toDouble());
        }
    }
    couponplugin->deleteLater();
    return true;
}

bool ProFeatures::CouponUpdate(int receiptNum, const QString &code, const QString &topay)
{
    if (code.isEmpty()) return false;

    IndependentInterface *couponplugin
        = qobject_cast<IndependentInterface *>(PluginManager::instance()->getObjectByName("Coupon"));
    if (couponplugin && couponplugin->isActivated()) {

        QMap<QString, QVariant> arguments;
        arguments.insert("whatever", 1);
        arguments.insert("receiptNumber", receiptNum);
        arguments.insert("topay", topay);
        arguments.insert("code", code);
        couponplugin->deleteLater();
        bool retval = couponplugin->process(arguments);
        return retval;
    }
    return true;
}

bool ProFeatures::isActive(const QString &pluginname)
{
    IndependentInterface *plugin
        = qobject_cast<IndependentInterface *>(PluginManager::instance()->getObjectByName(pluginname));
    if (plugin && plugin->isActivated()) {
        plugin->deleteLater();
        return true;
    }
    return false;
}

bool ProFeatures::updatePluginDatabase(const QString &name, const QString &table_name, int schema, const QString &path)
{
    QSqlDatabase dbc = Database::database(name.toUpper());
    int current_schema = schema;
    if (!dbc.tables(QSql::AllTables).contains(table_name)) {

        CSqlQuery query(dbc, Q_FUNC_INFO);
        QFile f;

        if (dbc.driverName() == "QSQLITE") {
            f.setFileName(QString("%1-sqlite.sql").arg(path));
        } else if (dbc.driverName() == "QMYSQL") {
            f.setFileName(QString("%1-mysql.sql").arg(path));
        }

        f.open(QIODevice::ReadOnly);
        QString cmd = f.readAll();
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
        QStringList commands = cmd.split(';', Qt::SkipEmptyParts);
#else
        QStringList commands = cmd.split(';', QString::SkipEmptyParts);
#endif
        foreach (const QString &command, commands) {
            if (command.trimmed().isEmpty()) continue;
            query.exec(command);
        }
        Database::insert2globals(QString("%1_schema").arg(name), 1, QVariant());
    } else { // db already exists; check if we need to run an update
        int schemaVersion = 1;
        QVariant value;
        QString strValue;
        int id = Database::select_globals(QString("%1_schema").arg(name), value, strValue);
        if (id > 0)
            schemaVersion = value.toInt();
        else // schemaVersion not set in globals, must be version 1
            Database::insert2globals(QString("%1_schema").arg(name), schemaVersion, QVariant());

        if (schemaVersion > current_schema) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << " false " << name << " schema version " << schemaVersion
                       << " current schema version " << current_schema;
            return false;
        }
        for (int i = schemaVersion + 1; i <= current_schema; i++) {
            QFile f;

            if (dbc.driverName() == "QSQLITE")
                f.setFileName(QString("%1-sqlite-update-%2.sql").arg(path).arg(i));
            else if (dbc.driverName() == "QMYSQL")
                f.setFileName(QString("%1-mysql-update-%2.sql").arg(path).arg(i));

            if (!f.open(QIODevice::ReadOnly)) {
                qCritical("could not load internal update file %s", qPrintable(f.fileName()));
                return false; // should never happen
            }

            QString cmd = f.readAll();
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
            QStringList commands = cmd.split(';', Qt::SkipEmptyParts);
#else
            QStringList commands = cmd.split(';', QString::SkipEmptyParts);
#endif
            bool ok;
            CSqlQuery query(dbc, Q_FUNC_INFO);
            foreach (const QString &command, commands) {
                if (command.trimmed().isEmpty()) continue;
                ok = query.exec(command);
                if (!ok) {
                    dbc.rollback();
                    return false;
                }
            }
            Database::insert2globals(QString("%1_schema").arg(name), i, QString());
        }

        if (schemaVersion != current_schema)
            Database::insert2globals(QString("%1_schema").arg(name), current_schema, QString());
    }

    return true;
}

void ProFeatures::processCouponPayment(
    CSqlQuery &query, int payedBy, QJsonObject &data, QMap<int, double> &given, int currentReceipt)
{

    qDebug() << "Function Name: " << Q_FUNC_INFO << " give " << given << " payedBy " << payedBy;
    if ((payedBy == PAYED_BY_CASH && given.value(PAYED_BY_CASH, 0.0) > 0.0)
        || (payedBy == PAYED_BY_COUPON && given.value(PAYED_BY_COUPON, 0.0) > 0.0)) {
        query.prepare(QString("INSERT INTO receiptspay (receiptNum, payedBy, "
                              "gross) VALUES (:receiptNum, :payedBy, :gross)"));
        query.bindValue(":receiptNum", currentReceipt);
        query.bindValue(":payedBy", payedBy);
        query.bindValue(":gross", given.value(payedBy));
        query.exec();
        if (given.value(PAYED_BY_DEBITCARD, 0.0) > 0.0) {
            data["secondPayText"] = tr("Bankomat");
            data["secondPayVal"] = given.value(PAYED_BY_DEBITCARD);
            query.bindValue(":payedBy", PAYED_BY_DEBITCARD);
            query.bindValue(":gross", given.value(PAYED_BY_DEBITCARD));
            query.exec();
        }
        if (given.value(PAYED_BY_CREDITCARD, 0.0) > 0.0) {
            data["secondPayText"] = tr("Kreditkarte");
            data["secondPayVal"] = given.value(PAYED_BY_CREDITCARD);
            query.bindValue(":payedBy", PAYED_BY_CREDITCARD);
            query.bindValue(":gross", given.value(PAYED_BY_CREDITCARD));
            query.exec();
        }
        if (payedBy != PAYED_BY_COUPON && given.value(PAYED_BY_COUPON, 0.0) > 0.0) {
            data["secondPayText"] = tr("Gutschein");
            data["secondPayVal"] = given.value(PAYED_BY_COUPON);
            query.bindValue(":payedBy", PAYED_BY_COUPON);
            query.bindValue(":gross", given.value(PAYED_BY_COUPON));
            query.exec();
        }
        if (payedBy == PAYED_BY_COUPON && given.value(PAYED_BY_CASH, 0.0) > 0.0) {
            data["secondPayText"] = tr("Bar");
            data["secondPayVal"] = given.value(PAYED_BY_CASH);
            query.bindValue(":payedBy", PAYED_BY_CASH);
            query.bindValue(":gross", given.value(PAYED_BY_CASH));
            query.exec();
        }
    }
}

bool ProFeatures::OptionalReciptPrint(QJsonObject &data)
{
    if (ProFeatures::isValid()) {
        ProFeaturesPrint fp;
        return fp.OptionalReciptPrint(data);
    }
    return true;
}
