/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include "utils.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "RK/rk_signaturemodule.h"
#include "RK/rk_signaturemodulefactory.h"
#include "backup.h"
#include "csqlquery.h"
#include "database.h"
#include "defines.h"
#include "demomode.h"
#include "qrcode.h"
#include "singleton/spreadsignal.h"

#include <QByteArray>
#include <QCollator>
#include <QDateTime>
#include <QDebug>
#include <QElapsedTimer>
#include <QFileInfo>
#include <QFont>
#include <QFontMetrics>
#include <QJsonObject>
#include <QPixmap>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlRecord>
#include <QStorageInfo>
#include <QStringList>
#include <QtMath>

Utils::Utils()
{
}

Utils::~Utils()
{
}

QString Utils::getSignature(QJsonObject data)
{

    Spread::Instance()->setProgressBarWait(true);
    RKSignatureModule *RKSignature = RKSignatureModuleFactory::createInstance("", DemoMode::isDemoMode());
    RKSignature->selectApplication();

    QString taxlocation = Database::getTaxLocation();
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QJsonObject sign;
    bool error = false;

    sign["SVI"] = RKSignature->getSVI();
    sign["Kassen-ID"] = data.value("kasse").toString();
    sign["Belegnummer"] = QString::number(data.value("receiptNum").toInt());
    sign["Beleg-Datum-Uhrzeit"] = data.value("receiptTime").toString();

    query.prepare("SELECT tax FROM taxtypes WHERE taxlocation=:taxlocation AND NOT comment='Satz-Erweitert' ORDER BY "
                  "id");
    query.bindValue(":taxlocation", taxlocation);

    query.setCriticalError(true);
    bool ok = query.exec();
    if (!ok) {
        error = true;
    }

    qlonglong counter = 0;
    while (query.next()) {
        double tax = query.value(0).toDouble();
        QBCMath gross = data.value(Database::getTaxType(tax)).toDouble();
        gross.round(2);

        sign[Database::getTaxType(tax)] = gross.toString();
        // counter += sign.value(Database::getTaxType( tax )).toString().toDouble()
        // * 100;
        counter += sign.value(Database::getTaxType(tax)).toString().replace(".", "").toLongLong();
    }

    QString concatenatedValue = sign["Kassen-ID"].toString() + sign["Belegnummer"].toString();
    QString lastUsedCertificateSerial = "";
    QString last_signature = getLastReceiptSignature();
    qlonglong turnOverCounter = getTurnOverCounter(RKSignature, lastUsedCertificateSerial, last_signature, error);

    turnOverCounter += counter;

    QString symmetricKey = RKSignature->getPrivateTurnoverKey();
    QString base64encryptedTurnOverCounter
        = RKSignature->encryptTurnoverCounter(concatenatedValue, turnOverCounter, symmetricKey);

    if (data.value("isStorno").toBool(false))
        sign["Stand-Umsatz-Zaehler-AES256-ICM"] = "U1RP";
    else
        sign["Stand-Umsatz-Zaehler-AES256-ICM"] = base64encryptedTurnOverCounter;

    bool safetyDevice;
    QString certificateSerial = RKSignature->getCertificateSerial(true);
    if (certificateSerial == "0" || certificateSerial.isEmpty()) {
        safetyDevice = false;
        certificateSerial = lastUsedCertificateSerial;
        if (!RKSignature->isSignatureModuleSetDamaged()) RKSignature->setSignatureModuleDamaged();
    } else {
        safetyDevice = true;
    }

    sign["Zertifikat-Seriennummer"] = certificateSerial;

    sign["Sig-Voriger-Beleg"] = RKSignature->getLastSignatureValue(last_signature);

    QElapsedTimer t;
    t.start();
    QString signature = RKSignature->signReceipt(getReceiptShortJson(sign));
    qDebug() << "Function Name: " << Q_FUNC_INFO << "Signature Time elapsed: " << t.elapsed() << " ms";

    delete RKSignature;

    Spread::Instance()->setProgressBarWait(false);
    Spread::Instance()->setSafetyDevice(safetyDevice);

    if (certificateSerial.isEmpty() || last_signature.isEmpty() || error) return "";

    return signature;
}

/*
return "_" + rkSuite.getSuiteID() + "_" + cashBoxID + "_" + receiptIdentifier.
+ "_" + dateFormat.format(receiptDateAndTime) + "_" +
decimalFormat.format(sumTaxSetNormal).
+ "_" + decimalFormat.format(sumTaxSetErmaessigt1) + "_" +
decimalFormat.format(sumTaxSetErmaessigt2).
+ "_" + decimalFormat.format(sumTaxSetNull) + "_" +
decimalFormat.format(sumTaxSetBesonders).
+ "_" + encryptedTurnoverValue + "_" + signatureCertificateSerialNumber + "_" +
signatureValuePreviousReceipt;
*/

QString Utils::getLastReceiptSignature()
{
    int lastReceiptId = Database::getLastReceiptNum() - 1;
    return getReceiptSignature(lastReceiptId, true);
}

QString Utils::getReceiptSignature(int id, bool full)
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << " id: " << id;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT data FROM dep WHERE receiptNum=:receiptNum");
    query.bindValue(":receiptNum", id);

    query.setCriticalError(true);
    bool ok = query.exec();
    if (!ok) {
        return QString();
    }

    if (query.next()) {
        QString s = query.value(0).toString();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " return: " << s;
        if (full) return s;

        return s.split('.').at(2);
    }

    return Database::getCashRegisterId();
}

QString Utils::getReceiptShortJson(QJsonObject sig)
{
    // RK Suite defined in Detailspezifikation/ABS 2
    // R1_AT100("1", "AT100", "ES256", "SHA-256", 8);

    QString sign = "_R1-";
    sign.append(sig.value("SVI").toString());
    sign.append("_");
    sign.append(sig.value("Kassen-ID").toString());
    sign.append("_");
    sign.append(sig.value("Belegnummer").toString());
    sign.append("_");
    sign.append(sig.value("Beleg-Datum-Uhrzeit").toString());
    sign.append("_");
    sign.append(sig.value("Satz-Normal").toString().replace(".", ","));
    sign.append("_");
    sign.append(sig.value("Satz-Ermaessigt-1").toString().replace(".", ","));
    sign.append("_");
    sign.append(sig.value("Satz-Ermaessigt-2").toString().replace(".", ","));
    sign.append("_");
    sign.append(sig.value("Satz-Null").toString().replace(".", ","));
    sign.append("_");
    sign.append(sig.value("Satz-Besonders").toString().replace(".", ","));
    sign.append("_");
    sign.append(sig.value("Stand-Umsatz-Zaehler-AES256-ICM").toString());
    sign.append("_");
    sign.append(sig.value("Zertifikat-Seriennummer").toString());
    sign.append("_");
    sign.append(sig.value("Sig-Voriger-Beleg").toString());

    qDebug() << "Function Name: " << Q_FUNC_INFO << " short: " << sign;

    return sign;
}

qlonglong Utils::getTurnOverCounter(
    RKSignatureModule *sm, QString &lastSerial, const QString &lastReceiptSignature, bool &error)
{
    QString key = RKSignatureModule::getPrivateTurnoverKey();

    if (lastReceiptSignature.isEmpty()) {
        error = true;
        return 0;
    }

    if (Database::getCashRegisterId() == lastReceiptSignature) return 0;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    //    query.prepare("SELECT data FROM dep");
    query.prepare("SELECT data FROM dep ORDER BY id DESC LIMIT 100");

    bool ok = query.exec();
    if (!ok) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        error = true;
        return 0;
    }

    qlonglong counter = 0;
    QString payload;
    if (query.first()) {
        payload = RKSignatureModule::base64Url_decode(query.value("data").toString().split('.').at(1));
        QStringList list = payload.split('_');
        QString encTOC = list.at(10);
        lastSerial = list.at(11);

        while (encTOC == "U1RP" && query.next()) {
            qlonglong counter2 = 0;
            for (int y = 5; y < 9; y++) {
                QString current = list.at(y);
                counter2 += current.replace(",", "").toLongLong();
            }
            counter += counter2;

            payload = RKSignatureModule::base64Url_decode(query.value("data").toString().split('.').at(1));
            list = payload.split('_');
            encTOC = list.at(10);
        }

        QString con = list.at(2) + list.at(3);
        QString decTOC = sm->decryptTurnoverCounter(con, encTOC, key);
        qlonglong TurnOverCounter = decTOC.toLongLong() + counter;
        return TurnOverCounter;
    }

    error = true;
    return 0;
}

double Utils::getYearlyTotal(int year)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QDateTime from;
    QDateTime to;

    if (year == 0) year = QDate::currentDate().year();

    QString fromString = QString("%1-01-01").arg(year);
    QString toString = QString("%1-12-31").arg(year);

    from.setDate(QDate::fromString(fromString, "yyyy-MM-dd"));
    to.setDate(QDate::fromString(toString, "yyyy-MM-dd"));
    to.setTime(QTime::fromString("23:59:59"));

    /* Summe */
    query.prepare("SELECT sum(gross) FROM receipts WHERE timestamp BETWEEN "
                  ":fromDate AND :toDate AND payedBy < :payedby");
    query.bindValue(":fromDate", from.toString(Qt::ISODate));
    query.bindValue(":toDate", to.toString(Qt::ISODate));
    query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

    query.exec();
    query.next();

    double sales = query.value(0).toDouble();

    return sales;
}

bool Utils::isDirectoryWritable(QString path)
{
    QFileInfo f(path);
    if (f.exists() && f.isDir()) {
        if (f.isWritable()) return true;
    }

    return false;
}

QString Utils::wordWrap(QString text, int width, QFont font)
{
    QFontMetrics fm(font);
    QString result;
    int space = 0;

    if (text.lastIndexOf(' ') < 0) {
        text.replace('/', " / ");
    }

    for (;;) {
        int i = 0;
        while (i < text.length()) {
#if (QT_VERSION >= QT_VERSION_CHECK(5, 11, 0))
            if (fm.horizontalAdvance(text.left(++i + 1)) > width) {
#else
            if (fm.width(text.left(++i + 1)) > width) {
#endif
                int j = text.indexOf('\n');
                if (j > i) {
                    if (j < i * 2) text.replace(j, 1, ' ');
                    j = -1;
                }
                if (j < 0) j = text.lastIndexOf(' ', i);

                space = 0;
                if (j >= 0) {
                    i = j;
                    space = 1;
                }
                result += text.left(i);
                result += '\n';
                text = text.mid(i + space);
                break;
            }
        }
        if (i >= text.length()) break;
    }
    return result + text;
}

bool Utils::checkTurnOverCounter(QStringList &error, QString &cardType)
{
    QString key = RKSignatureModule::getPrivateTurnoverKey();
    RKSignatureModule *sm = RKSignatureModuleFactory::createInstance("", DemoMode::isDemoMode());
    cardType = sm->getCardType();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    bool ret = true;

    if (Utils::getLastReceiptSignature().isEmpty()) {
        error.append(QObject::tr("Fehlerhaftes DEP-7 Tabellenschema. Keine DEP-7 Einträge gefunden."));
        ret = false;
    }

    query.prepare("SELECT id, receiptNum, data FROM dep ORDER BY id");
    query.exec();

    qlonglong counter = 0;
    QString lastused;
    while (query.next()) {
        QString data = query.value("data").toString();
        QString payload = RKSignatureModule::base64Url_decode(data.split('.').at(1));
        QStringList list = payload.split('_');
        qlonglong counter2 = 0;
        if (list.count() < 13) {
            qCritical() << "Function Name: " << Q_FUNC_INFO << " error: Faulty DEP-7 listCount: " << list.count();
            qCritical() << "Function Name: " << Q_FUNC_INFO << " error: payload: " << payload;
            return false;
        }

        QString con = list.at(2) + list.at(3);
        QString encTOC = list.at(10);
        QString serial = list.at(11);
        QString receiptNum = list.at(3);

        if (lastused.isNull()) {
            lastused = list.at(2);
        } else {
            if (lastused.compare(list.at(2)) != 0) {
                error.append(QObject::tr("Fehlerhaftes DEP-7 KassenId wurde "
                                         "manipuliert. BON %1 ('%2' != '%3')")
                        .arg(receiptNum)
                        .arg(list.at(2))
                        .arg(lastused));
                qCritical() << "Function Name: " << Q_FUNC_INFO
                            << " error: Faulty DEP-7 CashregisterId was manipulated. BON " << receiptNum;
            }
            lastused = list.at(2);
        }

        if (serial.isEmpty()) error.append(QObject::tr("Fehlende Seriennummer bei BON %1").arg(receiptNum));
        QString decTOC = sm->decryptTurnoverCounter(con, encTOC, key);
        for (int y = 5; y < 10; y++) {
            QString current = list.at(y);
            if (current.isEmpty())
                error.append(
                    QObject::tr("Fehlender Parameter im DEP bei BON %1, Parameter #%2").arg(receiptNum).arg(y));
            counter2 += current.replace(",", "").toLongLong();
        }
        counter += counter2;
        //        qDebug() << "#" << list.at(3) << "decTOC " << decTOC << " Counter
        //        " << counter << " L:" << list;
        QString newEncTOC = sm->encryptTurnoverCounter(con, counter, key);

        if (newEncTOC.compare(encTOC) != 0) {
            if (encTOC != "U1RP") {
                error.append(QObject::tr("Fehler beim Umsatzzähler für BON %1, Wert=%2 statt %3")
                        .arg(receiptNum)
                        .arg(decTOC)
                        .arg(counter));
                ret = false;
            }
        }
    }
    delete sm;
    ret = ret && error.size() == 0;
    return ret;
}

QString Utils::normalizeNumber(QString value)
{
    value.replace(".", ",");
    int pos = value.lastIndexOf(",");
    if (pos >= 0) value = value.left(pos) + "." + value.mid(pos + 1);

    return value.replace(",", "");
}

QString Utils::color_best_contrast(QString color)
{
    bool hash = false;
    bool ok;
    if (color.startsWith("#")) {
        color = color.replace("#", "");
        hash = true;
    }
    QString r, g, b;
    if (color.length() == 3) {
        QString tmp = "";
        tmp += color.mid(0, 1) + color.mid(0, 1);
        tmp += color.mid(2, 1) + color.mid(2, 1);
        tmp += color.mid(3, 1) + color.mid(3, 1);
        r = (tmp.mid(0, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
        g = (tmp.mid(2, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
        b = (tmp.mid(4, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
    } else if (color.length() == 6) {
        r = (color.mid(0, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
        g = (color.mid(2, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
        b = (color.mid(4, 2).toInt(&ok, 16) < 128) ? "FF" : "00";
    } else {
        r = "00";
        g = "00";
        b = "00";
    }

    return ((hash) ? "#" : "") + r + g + b;
}

QString Utils::taxRoundUp(double value, unsigned short np)
{
    double x, factor;
    QBCMath result;
    factor = pow(10, np);
    x = floor(value * factor + 0.9) * factor;
    result = floor(x / factor) / factor;
    return result.toString();
}

double Utils::getTax(double value, double tax, bool net)
{
    QBCMath v(value);
    v.round(2);
    QBCMath t(100 + tax);
    t.round(2);
    QBCMath result;

    if (net) {
        result = v / 100 * t;
        result -= v;
    } else {
        result = v / t * 100;
        result = v - result;
    }

    result.round(2);
    return result.toDouble();
}

double Utils::getNet(double gross, double tax)
{
    double g = QString::number(gross, 'f', 2).toDouble();
    double t = Utils::getTax(gross, tax);
    return QString::number(g - t, 'f', 2).toDouble();
}

double Utils::getGross(double net, double tax)
{
    double n = QString::number(net, 'f', 2).toDouble();
    double t = Utils::getTax(net, tax, true);
    return QString::number(n + t, 'f', 2).toDouble();
}

QPixmap Utils::getQRCode(int id, bool &isDamaged)
{
    if (id < 1) return QPixmap();

    isDamaged = false;

    QString qr_code_rep = "";
    QString signature = Utils::getReceiptSignature(id, true);
    if (signature.split('.').size() == 3) {
        qr_code_rep = signature.split('.').at(1);
        qr_code_rep = RKSignatureModule::base64Url_decode(qr_code_rep);
        qr_code_rep = qr_code_rep + "_" + RKSignatureModule::base64Url_decode(signature.split('.').at(2)).toBase64();
        if (signature.split('.').at(2) == RKSignatureModule::base64Url_encode("Sicherheitseinrichtung ausgefallen"))
            isDamaged = true;
    } else {
        isDamaged = true;
    }

    QRCode qr;
    QPixmap QR = qr.encodeTextToPixmap(qr_code_rep);

    return QR;
}

void Utils::diskSpace(QString path, qint64 &size, qint64 &bytesAvailable, double &percent)
{
    QStorageInfo storage = QStorageInfo::root();
    storage.setPath(path);

    if (storage.isReadOnly()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " storage rootpath: " << storage.rootPath();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " isReadOnly:" << storage.isReadOnly();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " name:" << storage.name();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " fileSystemType:" << storage.fileSystemType();
    }

    size = storage.bytesTotal() / 1024 / 1024;
    bytesAvailable = storage.bytesAvailable() / 1024 / 1024;

    qint64 usedspace = size - bytesAvailable;
    percent = (double(usedspace) / double(size));
    if (percent < 0) percent = 0;
    if (percent > 0.9) percent = 0.9;
}

bool Utils::isNumber(QVariant number)
{
    bool check;
    number.toDouble(&check);
    if (check) return check;

    QLocale().toDouble(number.toString(), &check);
    return check;
}

bool Utils::compareNames(const QString &s1, const QString &s2)
{
    QCollator collator;
    collator.setNumericMode(true);

    return collator.compare(s1, s2) < 0;
}

QString Utils::getTaxString(QBCMath tax, bool zero)
{
    QString taxPercent;
    if (Database::getTaxLocation() == "CH")
        taxPercent = QString("%1").arg(QString::number(tax.toDouble(), 'f', 2));
    else
        taxPercent = QString("%1").arg(tax.toInt());

    if (zero && taxPercent == "0") taxPercent = "00";

    return taxPercent;
}

int Utils::getDigitsFromString(QString string)
{
    QString digits = "";
    for (int i = 0; i < string.size(); i++) {
        if (string[i].isDigit()) digits.append(string[i]);
    }
    return digits.toInt();
}

void Utils::reorganisationProducts()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery renameQuery(dbc, Q_FUNC_INFO);
    CSqlQuery deleteQuery(dbc, Q_FUNC_INFO);

    dbc.transaction();
    query.exec("UPDATE products SET origin=id WHERE origin < (SELECT MIN(id) FROM products)");
    query.exec("UPDATE products SET itemnum=TRIM(itemnum), name=TRIM(name)");
    query.exec("UPDATE products SET origin=(SELECT MIN(id) FROM products WHERE "
               "origin=0) WHERE origin=0");

    query.prepare("select origin, version from products group by origin order by "
                  "origin, version");
    renameQuery.prepare("update products set visible=:visible where "
                        "version<:version and origin=:origin");
    deleteQuery.prepare("delete from products where visible=:visible and id=:id");
    deleteQuery.setShowError(false);

    query.exec();
    while (query.next()) {
        int origin = query.value("origin").toInt();
        int version = query.value("version").toInt();
        renameQuery.bindValue(":visible", -1);
        renameQuery.bindValue(":version", version);
        renameQuery.bindValue(":origin", origin);
        renameQuery.exec();
    }
    if (!dbc.commit()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " last dbc error: " << dbc.lastError().text();
        dbc.rollback();
    }

    /* Do not use transaction/commit for this case (FOREIGN KEY)
     * Try to delete unnecessary items.
     * In the event that an article cannot be deleted, it will be given the value
     * visible = -1 and is thus marked as deleted.
     */
    query.prepare("select id from products where visible=:visible");
    query.bindValue(":visible", -1);
    query.exec();
    while (query.next()) {
        deleteQuery.setShowError(false);
        deleteQuery.bindValue(":visible", -1);
        deleteQuery.bindValue(":id", query.value("id").toInt());
        deleteQuery.exec();
    }
}

void Utils::fixPowerFailure()
{
    QSqlDatabase dbc = Database::database();
    QString driverName = dbc.driverName();
    QVariant value;
    QString strValue;
    Database::select_globals("lastReceiptNum", value, strValue);
    bool backup = false;

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.exec("SELECT MAX(id), receiptNum FROM receipts");
    uint maxReceipt = 0;
    if (query.next()) maxReceipt = query.value(1).toUInt();

    uint maxDep = 0;
    if (RKSignatureModule::isDEPactive()) {
        query.exec("SELECT MAX(receiptNum) FROM dep");
        if (query.next()) maxDep = query.value(0).toUInt();

        if (value.toUInt() == maxReceipt && maxReceipt == maxDep) return;
        Backup::create();
        backup = true;
        query.prepare("DELETE FROM receipts WHERE receiptNum > :maxDep");
        query.bindValue(":maxDep", maxDep);
        query.exec();
    } else if (value.toUInt() == maxReceipt) {
        return;
    }

    if (!backup) Backup::create();

    query.exec("DELETE FROM receipts WHERE receiptNum IS NULL");
    if (driverName == "QMYSQL") {
        query.exec("ALTER TABLE receipts AUTO_INCREMENT = 0");
    } else {
        query.prepare("UPDATE sqlite_sequence SET seq = (SELECT MAX(id) FROM receipts) WHERE name=:tbl");
        query.bindValue(":tbl", "receipts");
        query.exec();
    }
    query.exec("SELECT MAX(id), receiptNum FROM receipts");
    if (query.next()) maxReceipt = query.value(1).toUInt();

    Database::updateGlobals("lastReceiptNum", QString::number(maxReceipt), QString());
}

void Utils::convertSeconds(int totalSeconds, int &hours, int &minutes, int &seconds)
{
    hours = totalSeconds / 3600;
    totalSeconds %= 3600;
    minutes = totalSeconds / 60;
    seconds = totalSeconds % 60;
}
