/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 */

#include "qrkgastrotableorder.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/texteditdialog.h"
#include "database.h"
#include "documentprinter.h"
#include "history.h"
#include "qrkgastro.h"
#include "qrkgastroorderdelegate.h"
#include "qrkgastroquickproduct.h"
#include "qrkgastrovoiddialog.h"
#include "qrkjournal.h"
#include "qrkprogress.h"
#include "qrksettings.h"
#include "ui_qrkgastrotableorder.h"

#include <QAction>
#include <QCompleter>
#include <QJsonArray>
#include <QJsonObject>
#include <QMenu>
#include <QMessageBox>
#include <QSqlError>

#include <QDebug>

QRKGastroTableOrder::QRKGastroTableOrder(QWidget *parent)
    : QWidget(parent)
    , ui(new Ui::QRKGastroTableOrder)
{

    ui->setupUi(this);
    ui->orderList->header()->setStretchLastSection(false);
    ui->orderList->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
    ui->orderList->header()->setSectionResizeMode(1, QHeaderView::Stretch);
    // ui->orderList->header()->setSectionResizeMode(2, QHeaderView::Stretch);
    ui->orderList->header()->setSectionResizeMode(2, QHeaderView::ResizeToContents);
    ui->orderList->header()->setSectionResizeMode(3, QHeaderView::ResizeToContents);
    ui->orderList->headerItem()->setTextAlignment(3, Qt::AlignCenter);

    QrkGastroOrderDelegate *delegate = new QrkGastroOrderDelegate(this);
    ui->orderList->setItemDelegateForColumn(2, delegate);

    //    ui->cancelButton->setHidden(true);

    connect(ui->orderList, &QTreeWidget::itemClicked, this, &QRKGastroTableOrder::itemClicked);

    m_history = new History(this);

    QGridLayout *layout = new QGridLayout();
    m_numericKeyPad = new NumericKeypad(true, this);
    layout->addWidget(m_numericKeyPad, 0, 0);
    layout->addItem(new QSpacerItem(1, 1, QSizePolicy::Expanding, QSizePolicy::Fixed), 0, 1);
    layout->setColumnStretch(1, 1);
    ui->gridLayout->addLayout(layout, ui->gridLayout->rowCount(), 0, 1, 3, Qt::AlignLeft);
    m_numericKeyPad->setHidden(false);
    m_numericKeyPad->setCountButtonHidden(true);
    m_numericKeyPad->setPriceButtonHidden(true);
    m_numericKeyPad->setDiscountButtonHidden(true);

    connect(ui->quickButtons, &QrkQuickButtons::addProductToOrderList, this,
        static_cast<void (QRKGastroTableOrder::*)(int)>(&QRKGastroTableOrder::addSelectedProduct));

    connect(ui->cancelButton, &QPushButton::clicked, this, &QRKGastroTableOrder::cancelSlot);
    connect(ui->doneButton, &QrkPushButton::clicked, this, &QRKGastroTableOrder::doneSlot);
    connect(ui->printButton, &QrkPushButton::clicked, this, &QRKGastroTableOrder::printSlot);
    connect(ui->payNowButton, &QrkPushButton::clicked, this, &QRKGastroTableOrder::payNowSlot);
    connect(m_numericKeyPad, &NumericKeypad::valueButtonPressed, this, &QRKGastroTableOrder::numPadValueButtonPressed);
    connect(m_numericKeyPad, &NumericKeypad::textChanged, [=]() {
        ui->numPadLabel->setText(m_numericKeyPad->text());
    });
    connect(ui->payExtrasCheckBox, &QCheckBox::toggled, [=]() {
        m_payExtras = ui->payExtrasCheckBox->isChecked();
    });
    connect(ui->withButton, &QPushButton::clicked, this, &QRKGastroTableOrder::withButtonSlot);
    connect(ui->withoutButton, &QPushButton::clicked, this, &QRKGastroTableOrder::withoutButtonSlot);
    connect(ui->plusButton, &QPushButton::clicked, this, &QRKGastroTableOrder::plusSlot);
    connect(ui->minusButton, &QPushButton::clicked, this, &QRKGastroTableOrder::minusSlot);
    connect(ui->removeButton, &QPushButton::clicked, this, &QRKGastroTableOrder::removeSlot);
    connect(ui->quickProductPushButton, &QPushButton::clicked, this, &QRKGastroTableOrder::quickProduct);

    QAction *payAction = new QAction(tr("Bezahlen"), this);
    QAction *payCashAction = new QAction(tr("Bar bezahlen"), this);

    QMenu *payMenu = new QMenu;
    payMenu->addAction(payAction);
    payMenu->addAction(payCashAction);

    ui->toolButton->setMenu(payMenu);
    ui->toolButton->setDefaultAction(payAction);

    payAction->setIcon(QIcon(":src/icons/pay.png"));
    payCashAction->setIcon(QIcon(":src/icons/money48.png"));

    // Verbinde das Signal mit einem Slot
    connect(ui->splitter, &QSplitter::splitterMoved, this, &QRKGastroTableOrder::adjustColumnVisibility);
    connect(payAction, &QAction::triggered, this,
        static_cast<void (QRKGastroTableOrder::*)()>(&QRKGastroTableOrder::payNowSlot));
    connect(payCashAction, &QAction::triggered, [=]() {
        m_paycash = true;
        payNowSlot();
        m_paycash = false;
    });

    refresh();
}

QRKGastroTableOrder::~QRKGastroTableOrder()
{
    writeSettings();
    delete ui;
}

void QRKGastroTableOrder::refresh()
{
    readSettings();
    emit ui->quickButtons->refresh();
    m_numericKeyPad->setSinglePriceButtonHidden(!RBAC::Instance()->hasPermission("gastro_user_can_change_singleprice"));
}

void QRKGastroTableOrder::numPadValueButtonPressed(const QString &text, NUMPAD_VALUE_BUTTON button)
{
    QModelIndex idx = ui->orderList->selectionModel()->currentIndex();
    if (idx.isValid()) {
        QBCMath newText(text);
        newText.round(2);

        QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

        if (selected.isEmpty() || selected[0]->parent()) return;

        if (button == NUMPAD_VALUE_BUTTON_SINGLE) {
            selected[0]->setData(2, QRKGastro::PRODUCT_PRICE, newText.toDouble());
            updateOrderSum();
        }
    }
}

void QRKGastroTableOrder::setTicketId(int id)
{
    m_currentTicket = id;
    ui->customerTextLineEdit->setText(QRKGastro::getGuestName(id));
    fillOrderList(ui->orderList, m_currentTicket);
    readSettings();
}

void QRKGastroTableOrder::setTableId(int id)
{
    m_currentTable = id;
    ui->infoLabel->setText(
        QString("%1 / %2").arg(QRKGastro::getRoomNameFromTableId(id)).arg(QRKGastro::getTableName(id)));
}

void QRKGastroTableOrder::addSelectedProduct(int product)
{
    addSelectedProduct(product, {});
}

void QRKGastroTableOrder::addSelectedProduct(int product, QJsonObject itemdata)
{

    if (product == 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    // if the with/without buttons are checked, we need to add this product as
    // sub-item to the currently selected one
    int extraType = -1;
    if (ui->withButton->isChecked())
        extraType = QRKGastro::TYPE_WITH;
    else if (ui->withoutButton->isChecked())
        extraType = QRKGastro::TYPE_WITHOUT;

    // make this a one-time action
    ui->withButton->setChecked(false);
    ui->withoutButton->setChecked(false);

    if (extraType != -1) {
        QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

        if (selected.isEmpty() || selected[0]->parent()) // do not add sub-sub item
            return;

        query.prepare(QString("SELECT name, gross, itemnum, tax FROM products WHERE id=:id"));
        query.bindValue(":id", product);

        if (!query.exec()) {
            return;
        }

        query.next();
        QBCMath singlegross(query.value("gross").toDouble());
        QBCMath singletax(query.value("tax").toDouble());
        singlegross.round(2);

        // Preis des aktuellen Hauptprodukts
        QBCMath currentGross(selected[0]->data(2, QRKGastro::PRODUCT_PRICE).toDouble());
        QBCMath currentTax(selected[0]->data(2, QRKGastro::PRODUCT_TAX).toDouble());

        int comp = QBCMath::bccomp(singletax.toString(), currentTax.toString());
        if (comp != 0) {
            showIncompatibleTaxRateMessage(this);
            return;
        }

        // find product in children and change type
        bool found = false;
        for (int i = 0; i < selected[0]->childCount(); i++) {
            QTreeWidgetItem *child = selected[0]->child(i);
            if (child->data(1, QRKGastro::PRODUCT_ID) == product) {
                // Wenn ein WITH-Extra existiert und wir ein WITHOUT-Extra hinzufügen
                if (extraType == QRKGastro::TYPE_WITHOUT
                    && child->data(1, QRKGastro::EXTRA_TYPE).toInt() == QRKGastro::TYPE_WITH) {
                    // Abziehen des WITH-Betrags
                    if (m_payExtras) {
                        currentGross -= singlegross.toDouble();
                        if (!child->isHidden()) currentGross -= singlegross.toDouble();
                        currentGross.round(2);
                        selected[0]->setData(2, QRKGastro::PRODUCT_PRICE, currentGross.toDouble());
                    }
                }
                // Wenn ein WITHOUT-Extra existiert und wir ein WITH-Extra hinzufügen
                else if (extraType == QRKGastro::TYPE_WITH
                    && child->data(1, QRKGastro::EXTRA_TYPE).toInt() == QRKGastro::TYPE_WITHOUT) {
                    // Abziehen des WITHOUT-Betrags
                    if (m_payExtras) {
                        currentGross += singlegross.toDouble(); // Hinzufügen des neuen WITH-Betrags
                        if (!child->isHidden())
                            currentGross += singlegross.toDouble(); // Hinzufügen des neuen WITH-Betrags

                        currentGross.round(2);
                        selected[0]->setData(2, QRKGastro::PRODUCT_PRICE, currentGross.toDouble());
                    }
                }

                child->setData(0, Qt::DisplayRole, (extraType == QRKGastro::TYPE_WITH) ? "+" : "-");
                child->setData(1, QRKGastro::EXTRA_TYPE, extraType);
                child->setData(2, QRKGastro::PRODUCT_PRICE, singlegross.toDouble());

                child->setHidden(false);

                found = true;
                m_history->historyInsertLine(tr("%1 EXTRA").arg((extraType == QRKGastro::TYPE_WITH) ? "+" : "-"),
                    tr("Artikel %1").arg(Database::getProductNameById(product)));
                break;
            }
        }

        if (!found) {
            QTreeWidgetItem *child = new QTreeWidgetItem(selected[0]);
            child->setData(1, QRKGastro::ORDER_ID, selected[0]->data(1, QRKGastro::ORDER_ID));

            child->setData(0, Qt::DisplayRole, (extraType == QRKGastro::TYPE_WITH) ? "+" : "-");
            child->setData(1, QRKGastro::EXTRA_TYPE, extraType);
            child->setData(1, Qt::DisplayRole, query.value("name").toString());
            child->setData(1, QRKGastro::PRODUCT_ID,
                product); // store unique product id
            child->setData(2, QRKGastro::PRODUCT_PRICE, singlegross.toDouble());

            if (m_payExtras) {
                QBCMath gross(selected[0]->data(2, QRKGastro::PRODUCT_PRICE).toDouble());
                (extraType == QRKGastro::TYPE_WITH) ? gross += singlegross : gross -= singlegross;
                gross.round(2);
                selected[0]->setData(2, QRKGastro::PRODUCT_PRICE, gross.toDouble());
            }

            selected[0]->setExpanded(true);
            m_history->historyInsertLine(tr("%1 EXTRA").arg((extraType == QRKGastro::TYPE_WITH) ? "+" : "-"),
                tr("Artikel %1").arg(Database::getProductNameById(product)));
        }

        updateOrderSum();
        return;
    }

    // find product in tree and update count, but only if this item does not
    // have already extra child-items defined
    bool found = false;
    for (int i = 0; i < ui->orderList->topLevelItemCount(); i++) {
        QTreeWidgetItem *item = ui->orderList->topLevelItem(i);
        if ((item->childCount() == 0) && (item->data(1, QRKGastro::PRODUCT_ID) == product)) {
            if (!itemdata.isEmpty()) break;
            int numpadvalue = ui->numPadLabel->text().toInt();
            if (numpadvalue == 0) numpadvalue = 1;
            if (item->isHidden()) {
                item->setData(0, Qt::DisplayRole, numpadvalue);
                item->setHidden(false);
            } else {
                item->setData(0, Qt::DisplayRole, item->data(0, Qt::DisplayRole).toInt() + numpadvalue);
            }

            emit ui->orderList->model()->dataChanged(
                ui->orderList->model()->index(i, 0), ui->orderList->model()->index(i, 2));

            m_numericKeyPad->clear();
            found = true;
            m_history->historyInsertLine(tr("%1 ARTIKEL").arg((extraType == QRKGastro::TYPE_WITH) ? "+" : "-"),
                tr("Artikel %1").arg(Database::getProductNameById(product)));
            break;
        }
    }

    if (!found) {
        query.prepare(QString("SELECT name, gross, tax FROM products WHERE id=:id"));
        query.bindValue(":id", product);
        bool ok = query.exec();
        if (!ok) {
            return;
        }

        query.next();

        int numpadvalue = ui->numPadLabel->text().toInt();
        if (numpadvalue == 0) numpadvalue = 1;

        QTreeWidgetItem *item = new QTreeWidgetItem;
        item->setData(0, Qt::DisplayRole, numpadvalue);
        item->setData(1, QRKGastro::PRODUCT_ID, product); // store unique product id
        item->setData(1, QRKGastro::ORDER_ID, -1);

        if (itemdata.isEmpty()) {
            item->setData(1, Qt::DisplayRole, query.value("name").toString());
            item->setData(2, QRKGastro::PRODUCT_PRICE, query.value("gross").toDouble());
            item->setData(2, QRKGastro::PRODUCT_TAX, query.value("tax").toDouble());
        } else {
            item->setData(1, Qt::DisplayRole, itemdata["name"].toString());
            item->setData(2, QRKGastro::PRODUCT_PRICE, itemdata["gross"].toDouble());
            item->setData(2, QRKGastro::PRODUCT_TAX, itemdata["tax"].toDouble());
        }

        item->setIcon(3, QIcon(":src/icons/textfield.png"));
        item->setSizeHint(0, QSize(50, 50));
        ui->orderList->addTopLevelItem(item);
        ui->orderList->scrollToBottom();
        ui->orderList->setCurrentItem(item);
        m_numericKeyPad->clear();
        m_history->historyInsertLine(tr("%1 ARTIKEL").arg((extraType == QRKGastro::TYPE_WITH) ? "+" : "-"),
            tr("Artikel %1").arg(Database::getProductNameById(product)));
    }

    updateOrderSum();
}

void QRKGastroTableOrder::showIncompatibleTaxRateMessage(QWidget *parent)
{
    QMessageBox::warning(parent, QObject::tr("Inkompatibler Artikel"),
        QObject::tr(
            "Der Artikel kann nicht als Extra hinzugefügt werden.\n"
            "Der Steuersatz des Artikels passt nicht zum Steuersatz des Hauptartikels. Bitte wählen Sie einen kompatiblen Artikel aus."),
        QMessageBox::Ok);
}

void QRKGastroTableOrder::itemClicked(QTreeWidgetItem *item, int column)
{

    if (item != Q_NULLPTR && item->parent() == Q_NULLPTR && column == 3) {
        QString data = item->data(column, QRKGastro::ORDER_DESCRIPTION).toString();
        TextEditDialog ted(this);
        ted.setText(data);
        if (ted.exec() == QDialog::Accepted) {
            data = ted.getText();
            item->setData(column, QRKGastro::ORDER_DESCRIPTION, data);
        }
    }
}

/*
void QRKGastroTableOrder::changeTicket(int tableId, int ticketId)
{
    ui->withButton->setChecked(false);
    ui->withoutButton->setChecked(false);
    //      clearGroupsLayout();

    fillOrderList(ui->orderList, m_currentTicket);

    updateOrderSum();
}
*/
void QRKGastroTableOrder::fillOrderList(QTreeWidget *tree, int ticketId)
{
    QRKGastro::fillOrderList(tree, ticketId);
    updateOrderSum();
}

void QRKGastroTableOrder::updateOrderSum()
{
    QBCMath sum(0);
    int totalVisibleChildCount = 0;

    int itemCount = ui->orderList->topLevelItemCount();
    for (int i = 0; i < itemCount; i++) {
        QTreeWidgetItem *item = ui->orderList->topLevelItem(i);

        if (item->isHidden()) {
            continue;
        }

        // Schleife durch die Kinder und nur sichtbare zählen
        int childCount = item->childCount();
        for (int j = 0; j < childCount; j++) {
            QTreeWidgetItem *childItem = item->child(j);
            if (!childItem->isHidden()) {
                totalVisibleChildCount++;
            }
        }

        // Berechne die Summe wie bisher
        sum += item->data(0, Qt::DisplayRole).toInt() * item->data(2, QRKGastro::PRODUCT_PRICE).toDouble();
    }

    sum.round(2);
    ui->sumLabel->setText(sum.toLocale() + " " + Database::getShortCurrency());

    qDebug() << "Function Name: " << Q_FUNC_INFO << " Total number of children: " << totalVisibleChildCount;

    ui->payExtrasCheckBox->setEnabled(!(totalVisibleChildCount > 0));
    ui->orderList->resizeColumnToContents(2);
    ui->orderList->header()->resizeSections(QHeaderView::ResizeToContents);
    emit updateOrderButton(m_currentTable);
}

void QRKGastroTableOrder::cancelSlot()
{
    doneSlot(false);
    emit cancelOrder(m_currentTable, true);
}

void QRKGastroTableOrder::payNowSlot()
{
    bool printorder = ui->printCheckBox->isChecked();
    doneSlot(printorder);
    emit payTicket(m_currentTicket, m_paycash);
}

void QRKGastroTableOrder::printSlot()
{
    doneSlot(true);
}

void QRKGastroTableOrder::doneSlot(bool printOrders)
{
    if (finishOrder()) {
        if (printOrders) printUnprintedOrders(m_currentTicket);
    }

    writeSettings();
    emit updateOrderButton(m_currentTable);
    emit cancelOrder(m_currentTable);
}

void QRKGastroTableOrder::printUnprintedOrders(int ticket)
{

    QRKProgress progress;
    progress.setText(tr("Bons werden gedruckt."));
    progress.setWaitMode(true);
    progress.show();
    qApp->processEvents();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT tableId FROM tickets WHERE id=:id");
    query.bindValue(":id", ticket);
    query.exec();
    query.next();
    int table = query.value("tableId").toInt();
    QString tableName
        = QString("%1 - %2").arg(QRKGastro::getRoomNameFromTableId(table)).arg(QRKGastro::getTableName(table));
    // print unprinted orders
    CSqlQuery orders(dbc, Q_FUNC_INFO);
    orders.prepare("SELECT (ticketorders.count - ticketorders.printed), products.name, "
                   "ticketorders.id, ticketorders.product FROM ticketorders "
                   "LEFT JOIN products ON ticketorders.product=products.id "
                   "WHERE ticketorders.ticketId=:ticketId AND (ticketorders.count > "
                   "ticketorders.printed)");
    orders.bindValue(":ticketId", ticket);
    orders.exec();

    CSqlQuery orderDescription(dbc, Q_FUNC_INFO);
    orderDescription.prepare("SELECT description FROM orderdescs WHERE orderId=:id AND type=1");

    QJsonObject data;
    if (orders.next()) {
        data["customerText"] = tr("Bestellung %1").arg(tableName);

        //        if ( printer.outputFormat() != QPrinter::PdfFormat )
        //        {
        //          painter.translate(WIDTH, printer.pageRect().height());
        //          painter.rotate(180);
        //        }

        do {
            QString description = "";
            int ordersId = orders.value("id").toInt();
            orderDescription.bindValue(":id", ordersId);
            orderDescription.exec();
            if (orderDescription.next()) description = orderDescription.value("description").toString();

            QJsonObject obj;
            QJsonArray arr;

            int printerid = Database::getPrinterIdFromProduct(orders.value("product").toInt());

            obj["product"] = orders.value(1).toString();
            obj["count"] = orders.value(0).toInt();
            if (!description.isEmpty()) obj["description"] = description;

            qDebug() << "Function Name: " << Q_FUNC_INFO << QString::number(orders.value(0).toInt());
            qDebug() << "Function Name: " << Q_FUNC_INFO << orders.value(1).toString();

            // check for orderextras
            CSqlQuery extras(dbc, Q_FUNC_INFO);
            extras.prepare("SELECT orderextras.type, products.name FROM orderextras "
                           " LEFT JOIN products ON orderextras.product=products.id"
                           " WHERE orderId=:orderId");
            extras.bindValue(":orderId", orders.value(2).toInt());
            extras.exec();

            QJsonArray extra;
            while (extras.next()) {
                extra.append(QString((extras.value(0).toInt() == QRKGastro::TYPE_WITH) ? "+" : "-") + ' '
                    + extras.value(1).toString());
                qDebug() << "Function Name: " << Q_FUNC_INFO
                         << QString((extras.value(0).toInt() == QRKGastro::TYPE_WITH) ? "+" : "-") + ' '
                        + extras.value(1).toString();
            }
            obj["extra"] = extra;
            arr = data[QString::number(printerid)].toArray();
            arr.append(obj);
            data[QString::number(printerid)] = arr;
        } while (orders.next());
        if (!data.isEmpty()) {
            DocumentPrinter p(this);
            p.printTagged(data);
        }
        //        ui->openTickets->refreshTickets();
    }
}

bool QRKGastroTableOrder::finishOrder()
{
    if (!QRKGastro::createOrUpdateTicket(
            ui->orderList, m_currentTicket, m_currentTable, ui->customerTextLineEdit->text()))
        return false;

    //  ui->openTickets->refreshTickets();

    return true;
}

void QRKGastroTableOrder::plusSlot()
{
    QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

    if (selected.isEmpty() || selected[0]->parent()) // only on toplevel items
        return;

    selected[0]->setData(0, Qt::DisplayRole, selected[0]->data(0, Qt::DisplayRole).toInt() + 1);
    m_history->historyInsertLine(
        tr("%1 ARTIKEL").arg("+"), tr("Artikel %1").arg(selected[0]->data(1, Qt::DisplayRole).toString()));

    updateOrderSum();
}

//--------------------------------------------------------------------------------
/*
void QRKGastroTableOrder::minusSlot()
{
    QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

    if (selected.isEmpty() || selected[0]->parent()) // only on toplevel items
        return;

    int selectedDataCount = selected[0]->data(0, Qt::DisplayRole).toInt();
    if (selected[0]->data(0, Qt::DisplayRole).toInt() > 1) {
        int realCount;
        int count = getCountOfProduct(selected, realCount);
        if (realCount - count < 1) {
            if (!voidDialog(selected[0]->data(1, Qt::DisplayRole).toString())) return;
        }
        selected[0]->setData(0, Qt::DisplayRole, selected[0]->data(0, Qt::DisplayRole).toInt() - 1);
        m_history->historyInsertLine(
            tr("%1 ARTIKEL").arg("-"), tr("Artikel %1").arg(selected[0]->data(1, Qt::DisplayRole).toString()));
    }

    updateOrderSum();
}
*/

void QRKGastroTableOrder::minusSlot()
{
    QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

    if (selected.isEmpty() || selected[0]->parent()) // nur auf oberster Ebene
        return;

    int selectedDataCount = selected[0]->data(0, Qt::DisplayRole).toInt();

    // Wenn die Anzahl in der Liste > 1 ist, kommen die Kriterien zur Anwendung
    if (selectedDataCount > 1) {
        int realCount;
        int unprintedCount = getCountOfProduct(selected, realCount); // Anzahl der nicht bonierten Artikel

        // `voidDialog` anzeigen, wenn `printed` gleich oder größer als `selectedDataCount` ist
        if (unprintedCount <= 0 && selectedDataCount <= realCount) {
            if (!voidDialog(selected[0]->data(1, Qt::DisplayRole).toString())) return;
        }

        // Zähler reduzieren
        selected[0]->setData(0, Qt::DisplayRole, selectedDataCount - 1);
        m_history->historyInsertLine(
            tr("%1 ARTIKEL").arg("-"), tr("Artikel %1").arg(selected[0]->data(1, Qt::DisplayRole).toString()));
    }

    updateOrderSum();
}

//--------------------------------------------------------------------------------

/*
void QRKGastroTableOrder::removeSlot()
{
    QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

    if (selected.isEmpty()) return;

    int realCount;
    int count = getCountOfProduct(selected, realCount);
    if (realCount - count < 1
        && selected[0]->parent() == Q_NULLPTR) { // Toplevel-Element (Bestellposition) wurde entfernt
        if (!voidDialog(selected[0]->data(1, Qt::DisplayRole).toString())) return;
    }

    QBCMath singlegross(selected[0]->data(2, QRKGastro::PRODUCT_PRICE).toDouble());
    singlegross.round(2);
    selected[0]->setHidden(true); // Markiere das Element als versteckt

    if (selected[0]->parent() == Q_NULLPTR) { // Toplevel-Element (Bestellposition) wurde entfernt
        m_history->historyInsertLine(
            tr("%1 ARTIKEL").arg("Gelöscht"), tr("Artikel %1").arg(selected[0]->data(1, Qt::DisplayRole).toString()));
        updateOrderSum();
        return;
    }

    // Hier den Preis des übergeordneten Elements anpassen
    QBCMath gross(selected[0]->parent()->data(2, QRKGastro::PRODUCT_PRICE).toDouble());

    // Überprüfen, ob es sich um ein WITH- oder WITHOUT-Extra handelt
    int extraType = selected[0]->data(1, QRKGastro::EXTRA_TYPE).toInt();

    // Preis anpassen, je nach Typ des Extras
    if (extraType == QRKGastro::TYPE_WITH) {
        // Bei einem WITH-Extra den Betrag abziehen
        gross -= singlegross;
    } else if (extraType == QRKGastro::TYPE_WITHOUT) {
        // Bei einem WITHOUT-Extra den Betrag hinzufügen
        gross += singlegross;
    }

    // Aktualisiere den Preis des übergeordneten Elements
    selected[0]->parent()->setData(2, QRKGastro::PRODUCT_PRICE, gross.toDouble());

    QRKGastro::createOrUpdateTicket(ui->orderList, m_currentTicket, m_currentTable, ui->customerTextLineEdit->text());
    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    orders.prepare("UPDATE ticketorders SET printed=printed -1 WHERE ticketId=:ticketId AND counter=printed");
    orders.bindValue(":ticketId", m_currentTicket);
    orders.exec();

    updateOrderSum();
}
*/

void QRKGastroTableOrder::removeSlot()
{
    QList<QTreeWidgetItem *> selected = ui->orderList->selectedItems();

    if (selected.isEmpty()) return;

    // Datenbankverbindung einrichten
    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    // Überprüfen, ob es sich um ein Toplevel-Element handelt
    if (selected[0]->parent() == Q_NULLPTR) { // Toplevel-Element (Bestellposition) wurde entfernt
        int totalPrinted = 0;
        int totalCount = 0;

        // Holen der ID für das ausgewählte Produkt
        int productId = selected[0]->data(1, QRKGastro::PRODUCT_ID).toInt();

        // Holen der orderId für die COUNT-Abfrage
        QVariant orderId_data = selected[0]->data(1, QRKGastro::ORDER_ID);
        int orderId = (orderId_data.isValid()) ? selected[0]->data(1, QRKGastro::ORDER_ID).toInt() : -1;

        // Wenn orderId -1 ist, bedeutet das, dass die Bestellung nicht boniert ist
        if (orderId == -1) {
            // Löschen der Kind-Elemente nur im Tree
            while (selected[0]->childCount() > 0) {
                QTreeWidgetItem *child = selected[0]->child(0); // Holen das erste Kind
                // child->setHidden(true);                         // Kind-Element löschen
                delete child;
            }
            // Das Toplevel-Element selbst löschen (nur aus dem Tree)
            selected[0]->setHidden(true);
        } else {
            // Hier handelt es sich um ein boniertes Toplevel-Element
            // Holen der Anzahl der gedruckten und totalen Artikel
            orders.prepare(
                "SELECT COUNT(*) AS totalCount, SUM(printed) AS totalPrinted FROM ticketorders WHERE ticketId = :ticketId AND product = :productId AND id = :orderId");
            orders.bindValue(":ticketId", m_currentTicket);
            orders.bindValue(":productId", productId);
            orders.bindValue(":orderId", orderId); // Verwendung der spezifischen orderId
            orders.exec();

            if (orders.next()) {
                totalCount = orders.value("totalCount").toInt();
                totalPrinted = orders.value("totalPrinted").toInt();
            }

            // Hier den voidDialog aufrufen, wenn alle Artikel gedruckt sind
            if (totalCount > 0 && totalCount == totalPrinted) {
                if (!voidDialog(selected[0]->data(1, Qt::DisplayRole).toString())) return;
            }

            // Löschen der Kind-Elemente
            while (selected[0]->childCount() > 0) {
                QTreeWidgetItem *child = selected[0]->child(0); // Holen das erste Kind

                // Hier wird die orderId für das Kind-Element ermittelt
                QVariant orderId_data = child->data(1, QRKGastro::ORDER_ID);
                int childOrderId = (orderId_data.isValid()) ? child->data(1, QRKGastro::ORDER_ID).toInt() : -1;

                // Löschen der Datenbankeinträge für die Kind-Elemente, wenn die orderId nicht -1 ist
                if (childOrderId > 0) {
                    orders.prepare(
                        "DELETE FROM orderextras WHERE ticketId = :ticketId AND product = :productId AND orderId = :orderId");
                    orders.bindValue(":ticketId", m_currentTicket);
                    orders.bindValue(":productId", child->data(1, QRKGastro::PRODUCT_ID).toInt());
                    orders.bindValue(":orderId", childOrderId); // Verwendung der spezifischen orderId
                    orders.exec();
                }

                // child->setHidden(true); // Kind-Element löschen
                delete child;
            }

            // Löschen des Eintrags in der ticketorders-Tabelle für das Toplevel-Element, wenn die orderId nicht -1 ist
            if (orderId > 0) {
                orders.prepare("DELETE FROM ticketorders WHERE ticketId = :ticketId AND id = :orderId");
                orders.bindValue(":ticketId", m_currentTicket);
                orders.bindValue(":orderId", orderId); // Verwendung der spezifischen orderId
                orders.exec();
            }

            // Danach löschen wir das Toplevel-Element selbst
            selected[0]->setHidden(true); // Lösche das Toplevel-Element
        }
    } else {
        // Hier handelt es sich um ein nicht-Toplevel-Element
        int extraType = selected[0]->data(1, QRKGastro::EXTRA_TYPE).toInt();
        QBCMath singlegross(selected[0]->data(2, QRKGastro::PRODUCT_PRICE).toDouble());
        singlegross.round(2);

        // Hier den Preis des übergeordneten Elements anpassen
        QBCMath gross(selected[0]->parent()->data(2, QRKGastro::PRODUCT_PRICE).toDouble());

        // Preis anpassen, je nach Typ des Extras
        if (extraType == QRKGastro::TYPE_WITH) {
            gross -= singlegross; // Bei einem WITH-Extra den Betrag abziehen
        } else if (extraType == QRKGastro::TYPE_WITHOUT) {
            gross += singlegross; // Bei einem WITHOUT-Extra den Betrag hinzufügen
        }

        // Aktualisiere den Preis des übergeordneten Elements
        selected[0]->parent()->setData(2, QRKGastro::PRODUCT_PRICE, gross.toDouble());

        QVariant orderId_data = selected[0]->parent()->data(1, QRKGastro::ORDER_ID);
        int orderId = (orderId_data.isValid()) ? orderId_data.toInt() : -1;
        QVariant productId_data = selected[0]->parent()->data(1, QRKGastro::PRODUCT_ID);
        int productId = (productId_data.isValid()) ? productId_data.toInt() : -1;

        int totalPrinted = 0;
        int totalCount = 0;

        orders.prepare(
            "SELECT COUNT(*) AS totalCount, SUM(printed) AS totalPrinted FROM ticketorders WHERE ticketId = :ticketId AND product = :productId AND id = :orderId");
        orders.bindValue(":ticketId", m_currentTicket);
        orders.bindValue(":productId", productId);
        orders.bindValue(":orderId", orderId); // Verwendung der spezifischen orderId
        orders.exec();

        if (orders.next()) {
            totalCount = orders.value("totalCount").toInt();
            totalPrinted = orders.value("totalPrinted").toInt();
        }

        // Hier den voidDialog aufrufen, wenn alle Artikel gedruckt sind
        if (totalCount > 0 && totalCount == totalPrinted) {
            if (!voidDialog(selected[0]->data(1, Qt::DisplayRole).toString())) return;
        }


        // Löschen aus der Datenbank für das aktuelle Element (Kind), wenn die orderId gültig ist
        if (orderId > 0) {
            orders.prepare(
                "DELETE FROM orderextras WHERE ticketId = :ticketId AND product = :productId AND orderId = :orderId");
            orders.bindValue(":ticketId", m_currentTicket);
            orders.bindValue(":productId", selected[0]->data(1, QRKGastro::PRODUCT_ID).toInt());
            orders.bindValue(":orderId", orderId); // Verwendung der spezifischen orderId
            orders.exec();
        }

        // Das aktuell ausgewählte Element löschen
        // selected[0]->setHidden(true);
        delete selected[0];
    }

    QRKGastro::createOrUpdateTicket(ui->orderList, m_currentTicket, m_currentTable, ui->customerTextLineEdit->text());

    updateOrderSum(); // Aktualisiere die Gesamtsumme
}

// Funktion, die die gedruckte Anzahl für die ORDER_ID abruft
int QRKGastroTableOrder::getPrintedCountForOrder(int orderId)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT printed FROM ticketorders WHERE id = :orderId");
    query.bindValue(":orderId", orderId);
    query.exec();

    if (query.next()) {
        return query.value(0).toInt(); // Rückgabe der gedruckten Anzahl
    }
    return 0; // Rückgabe 0, wenn nichts gefunden wurde
}

void QRKGastroTableOrder::withButtonSlot()
{
    if (ui->withoutButton->isChecked()) ui->withoutButton->setChecked(false);
}

// only one of "with" our "without" may be checked
void QRKGastroTableOrder::withoutButtonSlot()
{
    if (ui->withButton->isChecked()) ui->withButton->setChecked(false);
}

void QRKGastroTableOrder::quickProduct()
{
    QrkGastroQuickProduct qp(this);
    if (qp.exec() == QDialog::Accepted) {
        QJsonObject itemdata = qp.getProduct();
        int productId = itemdata["productId"].toInt();
        emit addSelectedProduct(productId, itemdata);
    }
}

/*
int QRKGastroTableOrder::getCountOfProduct(const QList<QTreeWidgetItem *> &selected, int &realCount)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);
    int count = selected[0]->data(0, Qt::DisplayRole).toInt();
    realCount = count;
    if (m_currentTicket <= 0) return 0;

    orders.prepare("SELECT (ticketorders.count - ticketorders.printed) AS count, "
                   "products.name, ticketorders.id FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:id AND (ticketorders.count >= "
                   "ticketorders.printed)");
    orders.bindValue(":id", m_currentTicket);
    orders.exec();
    if (orders.next()) count -= orders.value("count").toInt();

    return count;
}
*/

int QRKGastroTableOrder::getCountOfProduct(const QList<QTreeWidgetItem *> &selected, int &realCount)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);
    int count = selected[0]->data(0, Qt::DisplayRole).toInt();
    realCount = count; // Setze realCount auf die aktuelle Anzahl in der Liste

    // Wenn keine Ticket-ID vorhanden ist, gibt es keine bonierten Artikel, also "unprinted" gleich `realCount` setzen
    if (m_currentTicket <= 0) return realCount;

    // ID des Produkts über die Rolle `QRKGastro::PRODUCT_ID`
    int productId = selected[0]->data(1, QRKGastro::PRODUCT_ID).toInt();

    orders.prepare("SELECT ticketorders.count, ticketorders.printed FROM ticketorders "
                   "WHERE ticketorders.ticketId = :id AND ticketorders.product = :productId");
    orders.bindValue(":id", m_currentTicket);
    orders.bindValue(":productId", productId);
    orders.exec();

    int totalPrinted = 0;
    int totalCount = 0;

    if (orders.next()) {
        totalCount = orders.value("count").toInt();
        totalPrinted = orders.value("printed").toInt();
    }

    realCount = totalCount;           // Setze realCount auf die Anzahl in der DB
    return totalCount - totalPrinted; // Gib die Anzahl der nicht bonierten Artikel zurück
}

void QRKGastroTableOrder::writeSettings()
{
    QrkSettings settings;

    settings.beginGroup("Gastro");
    settings.save2Settings("splitterGeometry", ui->splitter->saveGeometry(), false);
    settings.save2Settings("splitterState", ui->splitter->saveState(), false);

    settings.endGroup();
}

void QRKGastroTableOrder::readSettings()
{

    QrkSettings settings;

    settings.beginGroup("Gastro");
    bool custompaybutton = settings.value("custompaybutton", false).toBool();
    ui->payNowButton->setHidden(custompaybutton);
    ui->toolButton->setVisible(custompaybutton);

    ui->printCheckBox->setChecked(settings.value("printorderatpaynow", false).toBool());
    ui->printCheckBox->setEnabled(true);
    m_payExtras = settings.value("payExtras", false).toBool();
    ui->payExtrasCheckBox->setChecked(m_payExtras);
    ui->proofs_checkBox->setChecked(settings.value("proofs_guestname", false).toBool());
    ui->orderList->setColumnHidden(
        ui->orderList->columnCount() - 1, !settings.value("orderdescription", true).toBool());
    ui->splitter->restoreGeometry(settings.value("splitterGeometry").toByteArray());
    ui->splitter->restoreState(settings.value("splitterState").toByteArray());
    settings.endGroup();

    adjustColumnVisibility();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT guestname FROM tickets GROUP BY guestname");
    query.exec();

    QStringList completerList;

    while (query.next()) {
        QString value = query.value("guestname").toString();
        completerList << value;
    }

    QCompleter *editorCompleter = new QCompleter(completerList);
    ui->customerTextLineEdit->setCompleter(editorCompleter);
}

void QRKGastroTableOrder::adjustColumnVisibility(int, int)
{
    int threshold = 350;
    // int size = ui->splitter->sizes().at(0);
    if (ui->splitter->sizes()[0] < threshold) {
        ui->orderList->setColumnHidden(2, true);
    } else {
        ui->orderList->setColumnHidden(2, false);
    }
}

bool QRKGastroTableOrder::voidDialog(const QString &text)
{
    QRKGastroVoidDialog dialog(this);
    QrkJournal journal;
    int rval = dialog.exec();
    if (rval < 1) return false;

    switch (rval) {
        case QRKGastro::CANCEL_FALSETALLIED:
            journal.journalInsertLine("STORNO", "Artikel wurde falsch boniert.");
            break;
        case QRKGastro::CANCEL_WAITTOLONG:
            journal.journalInsertLine("STORNO", "Wartezeit zu lange.");
            break;
        case QRKGastro::CANCEL_WARESPOILED:
            journal.journalInsertLine("STORNO", "Artikel verdorben.");
            break;
        case QRKGastro::CANCEL_CANCELLATION:
            journal.journalInsertLine("STORNO", "Storno.");
            break;
        case QRKGastro::CANCEL_OTHERGUESTWISH:
            journal.journalInsertLine("STORNO", "Anderer Gästewunsch.");
            break;
    }
    journal.journalInsertLine("TEXTPOSITION", text);

    return true;
}
